<?php
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');

// -------------------- تنظیمات پایه --------------------
mb_internal_encoding('UTF-8');
date_default_timezone_set('Asia/Tehran'); // اگر لازم بود تغییر بده

// اگر config.php داری، همین را require کن تا PDO و کانستنت‌ها از آن بیاید
require_once __DIR__ . '/../config.php'; // مسیرت را درست کن

// --- اتصال PDO ---
// اگر در config.php اتصال داری، از همان استفاده کن. در غیر این صورت نمونه‌ی زیر:
if (!isset($pdo)) {
    $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';
    try {
        $pdo = new PDO($dsn, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
    } catch (Throwable $e) {
        echo json_encode(['ok'=>0,'error'=>'DB connection failed']);
        exit;
    }
}

// -------------------- ورودی تاریخ --------------------
$date = $_GET['date'] ?? '';
if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) {
    echo json_encode(['ok'=>0,'error'=>'Invalid date']);
    exit;
}
$apiDate = str_replace('-', '', $date); // YYYYMMDD

// -------------------- گرفتن لیست تیم‌های منتخب --------------------
// فرض: جدول featured_teams با ستون team_id (INT PRIMARY KEY) 
// اگر جدول/نام ستونت فرق دارد، فقط این SELECT را تغییر بده.
try {
    $stmt = $pdo->query("SELECT team_id FROM featured_teams");
    $featured = $stmt->fetchAll(PDO::FETCH_COLUMN, 0);
    $featuredIds = array_map('intval', $featured);
    $featuredSet = array_fill_keys($featuredIds, true);
} catch (Throwable $e) {
    echo json_encode(['ok'=>0,'error'=>'Failed to load featured teams']);
    exit;
}
if (empty($featuredSet)) {
    echo json_encode(['ok'=>1,'html'=>'<div class="card p-4 text-[#ccc]">هیچ تیم منتخبی در دیتابیس نیست.</div>']);
    exit;
}

// -------------------- گرفتن داده از لایواسکور --------------------
$lsUrl = "https://prod-public-api.livescore.com/v1/api/app/date/soccer/{$apiDate}/3.30";
$ctx = stream_context_create(['http'=>['timeout'=>10, 'ignore_errors'=>true]]);
$json = @file_get_contents($lsUrl, false, $ctx);
if ($json === false) {
    echo json_encode(['ok'=>0,'error'=>'Fetch from LiveScore failed']);
    exit;
}
$data = json_decode($json, true);
if (!is_array($data) || empty($data['Stages'])) {
    echo json_encode(['ok'=>1,'html'=>'<div class="card p-4 text-[#ccc]">برای این تاریخ مسابقه‌ای یافت نشد.</div>']);
    exit;
}

// -------------------- ساخت خروجی HTML --------------------
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

// گروهبندی بر اساس لیگ/استیج
$groups = []; // Sid => ['title'=>..., 'rows'=>[...]]
foreach ($data['Stages'] as $stage) {
    $sid = (int)($stage['Sid'] ?? 0);
    $country = $stage['Cnm'] ?? '';
    $sname   = $stage['Snm'] ?? 'League';
    $title   = trim(($country ? $country.' / ' : '') . $sname);

    if (empty($stage['Events']) || !is_array($stage['Events'])) continue;

    foreach ($stage['Events'] as $ev) {
        // IDs
        $homeId = (int)($ev['T1'][0]['ID'] ?? 0);
        $awayId = (int)($ev['T2'][0]['ID'] ?? 0);

        // اگر هیچکدام جزو تیم‌های منتخب نبود، رد شو
        if (!isset($featuredSet[$homeId]) && !isset($featuredSet[$awayId])) continue;

        $home = $ev['T1'][0]['Nm'] ?? 'تیم میزبان';
        $away = $ev['T2'][0]['Nm'] ?? 'تیم مهمان';

        // زمان شروع: Esd = epoch ms (UTC)
        $kickMs = (int)($ev['Esd'] ?? 0);
        $kickTs = $kickMs > 0 ? (int)floor($kickMs/1000) : 0;
        $timeStr = $kickTs ? date('H:i', $kickTs) : '';

        // وضعیت مسابقه: Eps (Not Started/FT/HT/...)
        $status = $ev['Eps'] ?? '';

        // نتیجه فعلی اگر در جریان/پایان‌یافته است
        $scoreH = $ev['Tr1'] ?? null;
        $scoreA = $ev['Tr2'] ?? null;
        $score  = (is_numeric($scoreH) && is_numeric($scoreA)) ? "{$scoreH} - {$scoreA}" : '';

        $rowHtml = '<div class="match-row flex items-center justify-between border-b border-[#1f2632] last:border-0">'
                 .   '<div class="flex items-center gap-2">'
                 .     '<span class="text-[#e5e7eb]">'.h($home).'</span>'
                 .     '<span class="text-[#aab5c6]">vs</span>'
                 .     '<span class="text-[#e5e7eb]">'.h($away).'</span>'
                 .   '</div>'
                 .   '<div class="flex items-center gap-3">'
                 .     ($score ? '<span class="text-[#ccc] text-sm">'.$score.'</span>' : '')
                 .     ($status ? '<span class="text-xs px-2 py-1 rounded bg-[#2a2f3a] text-[#aab5c6]">'.h($status).'</span>' : '')
                 .     ($timeStr ? '<span class="text-[#aab5c6]">'.$timeStr.'</span>' : '')
                 .   '</div>'
                 . '</div>';

        if (!isset($groups[$sid])) {
            $groups[$sid] = ['title'=>$title, 'rows'=>[]];
        }
        $groups[$sid]['rows'][] = $rowHtml;
    }
}

// اگر هیچ مسابقه‌ای مطابق نبود
if (empty($groups)) {
    echo json_encode(['ok'=>1,'html'=>'<div class="card p-4 text-[#ccc]">در این تاریخ هیچ مسابقه‌ای برای تیم‌های منتخب یافت نشد.</div>']);
    exit;
}

// مونتاژ HTML نهایی با حفظ استایل شما
$out = '';
foreach ($groups as $g) {
    $out .= '<div class="league-box mb-5">'
          .   '<div class="league-header px-3 py-2">'
          .     '<h3 class="text-[#4CAF50] font-bold">🏆 '.h($g['title']).'</h3>'
          .   '</div>';
    foreach ($g['rows'] as $row) $out .= $row;
    $out .= '</div>';
}

echo json_encode(['ok'=>1,'html'=>$out]);
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
  <title>تقویم بازی‌ها</title>
  <script src="https://cdn.tailwindcss.com"></script>

  <!-- تقویم شمسی -->
  <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/persian-datepicker@1.2.0/dist/css/persian-datepicker.min.css">
  <script src="https://cdn.jsdelivr.net/npm/jquery@3.7.1/dist/jquery.min.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/persian-date@1.1.0/dist/persian-date.js"></script>
  <script src="https://cdn.jsdelivr.net/npm/persian-datepicker@1.2.0/dist/js/persian-datepicker.min.js"></script>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Bai+Jamjuree:wght@200;300;400;500;600;700&amp;family=Vazirmatn:wght@100..900&amp;display=swap" rel="stylesheet">

  <style>
  :root{ --bg1:#0b0f16; --border:rgba(255,255,255,.10); --text:#e8eef7; --muted:#aab5c6; --accent:#4CAF50; }
  body{
    margin:0; padding:16px 16px calc(16px + env(safe-area-inset-bottom) + 72px); min-height:100dvh;
    font-family:'Bai Jamjuree','Vazirmatn',sans-serif; color:var(--text);
    background:
      radial-gradient(900px 500px at 110% -20%, rgba(255,122,26,.10), transparent 60%),
      radial-gradient(700px 350px at -20% 120%, rgba(60,130,246,.10), transparent 60%),
      linear-gradient(180deg,#0a111a 0%, var(--bg1) 30%, #0a1118 100%);
  }
  .container{max-width:820px;margin:0 auto}
  .card{
    background:linear-gradient(180deg, rgba(255,255,255,.06), rgba(255,255,255,.03));
    border:1px solid var(--border); border-radius:12px;
    box-shadow:0 10px 30px rgba(0,0,0,.25); backdrop-filter: blur(6px);
  }
  .header-bar{display:flex;align-items:center;justify-content:space-between;direction:ltr}
  .brand{
    background:linear-gradient(90deg,#2E7D32 0%,#4CAF50 40%,#B2FF59 75%,#E8FFF7 100%);
    -webkit-background-clip:text; background-clip:text; color:transparent;
    filter: drop-shadow(0 2px 10px rgba(76,175,80,.25));
  }
  a.btn{ padding:.45rem .7rem; border-radius:8px; background:#4CAF50; color:#000 !important; text-decoration:none; }
    
    .league-box{
      background:#171c24; border:1px solid #2a2f3a; border-radius:6px;
      overflow:hidden; box-shadow:0 2px 8px rgba(0,0,0,.25);
    }
    .league-header{ background:#030903; border-bottom:1px solid #1f2632; }
    .match-row{ padding:.75rem 1rem; }
    .match-row:hover{ background:rgba(76, 175, 80, 0.06); }
    
    /* Bottom Navigation */
.bottom-nav{
  position:fixed; left:0; right:0; bottom:0; z-index:50;
  background:rgba(10,17,26,.85);
  border-top:1px solid var(--border);
  backdrop-filter: blur(8px);
}
.bottom-nav .wrap{display:flex; justify-content:space-between; gap:4px}
.bottom-nav a.item{
  flex:1; padding:8px 0 10px; text-align:center; font-size:12px;
  color:var(--muted); text-decoration:none;
  display:flex; flex-direction:column; align-items:center; gap:4px;
}
.bottom-nav a.item i{font-size:18px}
.bottom-nav a.item.active{color:#4CAF50}
.bottom-nav .safe{height:env(safe-area-inset-bottom)}

    @media (max-width: 640px) {
      input[type="date"]::-webkit-calendar-picker-indicator { filter: invert(1); }
    }
    .mode-btn { padding: .5rem .75rem; border-radius: .5rem; }
    .mode-active { background:#4CAF50; color:#000; font-weight:600; }
    .mode-idle { background:#2a2a2a; color:#e5e7eb; }
  </style>
</head>

<body>
  <div class="container">
    <!-- Header -->
    <header class="card header-bar px-4 py-3">
        <h1 class="text-xl md:text-2xl brand font-extrabold">LiveCube</h1>
    </header>

<section class="card px-4 py-4 mb-6 mt-4">
  <div class="flex items-center justify-between gap-3 mb-5">
    <h2 class="text-xl md:text-2xl font-bold text-[#ccc]">تقویم بازی‌ها</h2>
    <div class="flex gap-2">
      <button id="btnFa" class="mode-btn mode-active">شمسی</button>
      <button id="btnEn" class="mode-btn mode-idle">میلادی</button>
    </div>
  </div>

  <div class="mx-auto w-full max-w-2xl flex flex-col sm:flex-row items-stretch sm:items-end justify-center gap-3">
    <!-- شمسی -->
    <input type="text" id="matchDateFa"
           class="border border-gray-700 bg-[#0a0a12] text-gray-300 rounded-lg p-2 w-full sm:w-64"
           placeholder="تاریخ شمسی" />
    <!-- میلادی -->
    <input type="date" id="matchDateG"
           class="border border-gray-700 bg-[#0a0a12] text-gray-300 rounded-lg p-2 w-full sm:w-64 focus:ring-2 focus:ring-orange-400 focus:outline-none"
           style="display:none" />
    <!-- خروجی میلادی از تقویم شمسی -->
    <input type="hidden" id="matchDate" />

    <button onclick="loadMatches()"
            class="bg-[#ccc] hover:bg-[#4CAF50] text-black font-medium rounded-lg transition-colors h-11 px-6 sm:px-8 min-w-[140px]">
      نمایش
    </button>
  </div>
</section>

    <div id="matchResults" class="mt-6"></div>
  </div>

<script>
$(document).ready(function () {
  const gToday = new Date().toISOString().split('T')[0];
  const pToday = new persianDate().format('YYYY/MM/DD');

  // فعال کردن تقویم شمسی
  $("#matchDateFa").persianDatepicker({
    initialValue: true,
    format: 'YYYY/MM/DD',
    autoClose: true,
    calendar: { persian: { locale: 'fa' } },
    onSelect: function (unix) {
      // تبدیل به میلادی و ذخیره در hidden
      const gDate = new persianDate(unix).toGregorian();
      const gStr = `${gDate.year}-${String(gDate.month).padStart(2, '0')}-${String(gDate.day).padStart(2, '0')}`;
      $("#matchDate").val(gStr);
    }
  });

  // پیش‌فرض: امروز
  $("#matchDateFa").val(pToday);
  $("#matchDate").val(gToday);
  $("#matchDateG").val(gToday);

  // دکمه شمسی
  $("#btnFa").click(function () {
    $(this).addClass('mode-active').removeClass('mode-idle');
    $("#btnEn").addClass('mode-idle').removeClass('mode-active');
    $("#matchDateFa").show();
    $("#matchDateG").hide();
    // همگام‌سازی تاریخ
    $("#matchDate").val(
      new persianDate().parse($("#matchDateFa").val()).toGregorian().year + '-' +
      String(new persianDate().parse($("#matchDateFa").val()).toGregorian().month).padStart(2, '0') + '-' +
      String(new persianDate().parse($("#matchDateFa").val()).toGregorian().day).padStart(2, '0')
    );
  });

  // دکمه میلادی
  $("#btnEn").click(function () {
    $(this).addClass('mode-active').removeClass('mode-idle');
    $("#btnFa").addClass('mode-idle').removeClass('mode-active');
    $("#matchDateFa").hide();
    $("#matchDateG").show();
    // همگام‌سازی تاریخ
    $("#matchDate").val($("#matchDateG").val());
  });

  // تغییر در ورودی میلادی
  $("#matchDateG").on("change", function () {
    $("#matchDate").val($(this).val());
  });
});

  async function loadMatches() {
    const gDate = $("#matchDate").val(); // باید YYYY-MM-DD باشد
    if (!gDate || !/^\d{4}-\d{2}-\d{2}$/.test(gDate)) {
      alert("تاریخ معتبر نیست");
      return;
    }

    const $box = $("#matchResults");
    $box.html('<div class="card p-4 text-[#aab5c6]">در حال دریافت...</div>');

    try {
      const res = await fetch('/LiveScore/api/featured_by_date.php?date=' + encodeURIComponent(gDate), {
        headers: { 'Accept': 'application/json' }
      });
      const data = await res.json();
      if (data.ok) {
        $box.html(data.html || '<div class="card p-4 text-[#ccc]">موردی یافت نشد.</div>');
      } else {
        $box.html('<div class="card p-4 text-[#fca5a5]">خطا: ' + (data.error || 'نامشخص') + '</div>');
      }
    } catch (e) {
      $box.html('<div class="card p-4 text-[#fca5a5]">خطا در ارتباط با سرور.</div>');
    }
  }
</script>
    
    <?php $active='calendar'; include __DIR__.'/parts/bottom-nav.php'; ?>

</body>
</html>